#!/bin/bash

set -e

echo "========================================"
echo "  Cloudlinker Installer for Linux"
echo "========================================"
echo

# Default values
INSTALL_DIR="/opt/cloudlinker"
ADD_AUTOSTART=false
CREATE_SERVICE=false

# Parse arguments
while [[ $# -gt 0 ]]; do
    case $1 in
        --dir)
            INSTALL_DIR="$2"
            shift 2
            ;;
        --autostart)
            ADD_AUTOSTART=true
            shift
            ;;
        --service)
            CREATE_SERVICE=true
            shift
            ;;
        --help)
            echo "Usage: sudo $0 [OPTIONS]"
            echo
            echo "Options:"
            echo "  --dir DIR      Install to specified directory (default: /opt/cloudlinker)"
            echo "  --autostart    Add to desktop autostart (for GUI sessions)"
            echo "  --service      Create systemd user service (for headless/server use)"
            echo "  --help         Show this help message"
            echo
            echo "Note: Installation to /opt requires root privileges (use sudo)."
            exit 0
            ;;
        *)
            echo "Unknown option: $1"
            exit 1
            ;;
    esac
done

echo "Installation directory: $INSTALL_DIR"
echo "Desktop autostart: $ADD_AUTOSTART"
echo "Systemd service: $CREATE_SERVICE"
echo

# Check if we need root privileges for /opt
if [[ "$INSTALL_DIR" == /opt/* ]] && [ "$EUID" -ne 0 ]; then
    echo "ERROR: Installation to $INSTALL_DIR requires root privileges."
    echo "Please run: sudo $0 $@"
    exit 1
fi

# Minimum required Java version
MIN_JAVA_VER=21

# Check Java installation
echo "Checking Java installation..."
if ! command -v java &> /dev/null; then
    echo "ERROR: Java is not installed."
    echo "Please install Java $MIN_JAVA_VER or higher."
    exit 1
fi

# Extract Java version (handles both "21.0.1" and "1.8.0" formats)
JAVA_VER_RAW=$(java -version 2>&1 | head -n 1 | cut -d'"' -f2)
JAVA_MAJOR=$(echo "$JAVA_VER_RAW" | cut -d'.' -f1)

# Handle old "1.x" version format (e.g., 1.8.0 = Java 8)
if [ "$JAVA_MAJOR" = "1" ]; then
    JAVA_MAJOR=$(echo "$JAVA_VER_RAW" | cut -d'.' -f2)
fi

echo "Found Java version: $JAVA_VER_RAW (major: $JAVA_MAJOR)"

# Check if version meets minimum requirement
if [ "$JAVA_MAJOR" -lt "$MIN_JAVA_VER" ]; then
    echo
    echo "ERROR: Java $JAVA_MAJOR is too old. Cloudlinker requires Java $MIN_JAVA_VER or higher."
    exit 1
fi
echo "Java version OK."

# Create installation directory
echo
echo "Creating installation directory..."
mkdir -p "$INSTALL_DIR"

# Get script directory
SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"

# Copy files
echo "Copying files..."
cp -f "$SCRIPT_DIR/cloudlinker-1.0.jar" "$INSTALL_DIR/"
cp -f "$SCRIPT_DIR/lets-encrypt-root.crt" "$INSTALL_DIR/"
cp -f "$SCRIPT_DIR/isrgrootx1.pem" "$INSTALL_DIR/"
cp -f "$SCRIPT_DIR/cloudlinker.ico" "$INSTALL_DIR/"
cp -f "$SCRIPT_DIR/cloudlinker.png" "$INSTALL_DIR/"

# Create config.properties if it doesn't exist
if [ ! -f "$INSTALL_DIR/config.properties" ]; then
    echo "Creating default configuration file..."
    cp "$SCRIPT_DIR/config.properties.template" "$INSTALL_DIR/config.properties"
    echo
    echo "IMPORTANT: Please edit $INSTALL_DIR/config.properties"
    echo "and fill in your organization_id and api_key."
fi

# Create launcher script
echo "Creating launcher script..."
cat > "$INSTALL_DIR/cloudlinker.sh" << 'LAUNCHER'
#!/bin/bash
cd "$(dirname "$0")"
java -jar cloudlinker-1.0.jar "$@"
LAUNCHER
chmod +x "$INSTALL_DIR/cloudlinker.sh"

# Create symlink in ~/.local/bin
echo "Creating symlink in ~/.local/bin..."
mkdir -p "$HOME/.local/bin"
ln -sf "$INSTALL_DIR/cloudlinker.sh" "$HOME/.local/bin/cloudlinker"

# Create desktop entry
echo "Creating desktop entry..."
mkdir -p "$HOME/.local/share/applications"
cat > "$HOME/.local/share/applications/cloudlinker.desktop" << DESKTOP
[Desktop Entry]
Name=Cloudlinker
Comment=Cloud Printing Client
Exec=$INSTALL_DIR/cloudlinker.sh
Icon=$INSTALL_DIR/cloudlinker.ico
Terminal=false
Type=Application
Categories=Utility;
DESKTOP

# Add to autostart if requested
if [ "$ADD_AUTOSTART" = true ]; then
    echo "Adding to desktop autostart..."
    mkdir -p "$HOME/.config/autostart"
    cat > "$HOME/.config/autostart/cloudlinker.desktop" << AUTOSTART
[Desktop Entry]
Name=Cloudlinker
Comment=Cloud Printing Client
Exec=$INSTALL_DIR/cloudlinker.sh
Icon=$INSTALL_DIR/cloudlinker.ico
Terminal=false
Type=Application
Categories=Utility;
X-GNOME-Autostart-enabled=true
AUTOSTART
    echo "Cloudlinker will start automatically on login."
fi

# Create systemd user service if requested
if [ "$CREATE_SERVICE" = true ]; then
    echo "Creating systemd user service..."
    mkdir -p "$HOME/.config/systemd/user"
    cat > "$HOME/.config/systemd/user/cloudlinker.service" << SERVICE
[Unit]
Description=Cloudlinker Cloud Printing Client
After=network-online.target
Wants=network-online.target

[Service]
Type=simple
WorkingDirectory=$INSTALL_DIR
ExecStart=/usr/bin/java -jar $INSTALL_DIR/cloudlinker-1.0.jar
Restart=on-failure
RestartSec=10

[Install]
WantedBy=default.target
SERVICE

    # Reload systemd and enable service
    systemctl --user daemon-reload
    systemctl --user enable cloudlinker.service
    echo "Systemd service created and enabled."
    echo "Start with: systemctl --user start cloudlinker"
    echo "View logs: journalctl --user -u cloudlinker -f"
fi

echo
echo "========================================"
echo "  Installation Complete!"
echo "========================================"
echo
echo "Cloudlinker has been installed to: $INSTALL_DIR"
echo
echo "To start Cloudlinker:"
echo "  - Run: cloudlinker (if ~/.local/bin is in PATH)"
echo "  - Or: $INSTALL_DIR/cloudlinker.sh"
echo
echo "To configure, edit: $INSTALL_DIR/config.properties"
echo
if [ "$ADD_AUTOSTART" = false ] && [ "$CREATE_SERVICE" = false ]; then
    echo "TIP: Re-run with --autostart for GUI autostart"
    echo "     or --service for headless systemd service."
    echo
fi
